<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

try {
    // Carregar configuração do banco
    $configFile = '../config/database.json';
    if (!file_exists($configFile)) {
        throw new Exception('Configuração do banco não encontrada');
    }
    
    $config = json_decode(file_get_contents($configFile), true);
    if (!$config) {
        throw new Exception('Erro ao ler configuração do banco');
    }
    
    // Conectar ao banco
    $pdo = new PDO(
        "mysql:host={$config['host']};dbname={$config['dbname']};charset=utf8mb4",
        $config['username'],
        $config['password'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]
    );
    
    // Gerar relatório
    $report = gerarRelatorio($pdo);
    
    echo json_encode([
        'success' => true,
        'report' => $report
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

function gerarRelatorio($pdo) {
    $html = '<div class="report-container">';
    
    // Estatísticas gerais
    $html .= '<div class="report-section">';
    $html .= '<h3><i class="fas fa-chart-bar"></i> Estatísticas Gerais</h3>';
    
    // Total de associados
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM Associados");
    $totalAssociados = $stmt->fetch()['total'];
    
    // Por situação
    $stmt = $pdo->query("SELECT situacao, COUNT(*) as total FROM Associados WHERE situacao IS NOT NULL GROUP BY situacao ORDER BY total DESC");
    $porSituacao = $stmt->fetchAll();
    
    // Por corporação
    $stmt = $pdo->query("
        SELECT m.corporacao, COUNT(*) as total 
        FROM Associados a 
        INNER JOIN Militar m ON a.id = m.associado_id 
        WHERE m.corporacao IS NOT NULL 
        GROUP BY m.corporacao 
        ORDER BY total DESC
    ");
    $porCorporacao = $stmt->fetchAll();
    
    $html .= '<div class="stats-grid">';
    $html .= '<div class="stat-card">';
    $html .= '<div class="stat-number">' . number_format($totalAssociados) . '</div>';
    $html .= '<div class="stat-label">Total de Associados</div>';
    $html .= '</div>';
    
    // Associados com endereço
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM Associados a INNER JOIN Endereco e ON a.id = e.associado_id");
    $comEndereco = $stmt->fetch()['total'];
    $html .= '<div class="stat-card">';
    $html .= '<div class="stat-number">' . number_format($comEndereco) . '</div>';
    $html .= '<div class="stat-label">Com Endereço</div>';
    $html .= '<div class="stat-percent">' . round(($comEndereco/$totalAssociados)*100, 1) . '%</div>';
    $html .= '</div>';
    
    // Associados com dados militares
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM Associados a INNER JOIN Militar m ON a.id = m.associado_id");
    $comMilitar = $stmt->fetch()['total'];
    $html .= '<div class="stat-card">';
    $html .= '<div class="stat-number">' . number_format($comMilitar) . '</div>';
    $html .= '<div class="stat-label">Com Dados Militares</div>';
    $html .= '<div class="stat-percent">' . round(($comMilitar/$totalAssociados)*100, 1) . '%</div>';
    $html .= '</div>';
    
    // Redes sociais
    $stmt = $pdo->query("SELECT COUNT(DISTINCT associado_id) as total FROM Redes_sociais");
    $comRedes = $stmt->fetch()['total'];
    $html .= '<div class="stat-card">';
    $html .= '<div class="stat-number">' . number_format($comRedes) . '</div>';
    $html .= '<div class="stat-label">Com Redes Sociais</div>';
    $html .= '<div class="stat-percent">' . round(($comRedes/$totalAssociados)*100, 1) . '%</div>';
    $html .= '</div>';
    
    $html .= '</div>'; // stats-grid
    $html .= '</div>'; // report-section
    
    // Distribuição por situação
    if (!empty($porSituacao)) {
        $html .= '<div class="report-section">';
        $html .= '<h3><i class="fas fa-users"></i> Distribuição por Situação</h3>';
        $html .= '<div class="chart-container">';
        foreach ($porSituacao as $sit) {
            $percent = round(($sit['total']/$totalAssociados)*100, 1);
            $html .= '<div class="chart-bar">';
            $html .= '<div class="chart-label">' . ucfirst($sit['situacao']) . '</div>';
            $html .= '<div class="chart-progress">';
            $html .= '<div class="chart-fill" style="width: ' . $percent . '%"></div>';
            $html .= '</div>';
            $html .= '<div class="chart-value">' . $sit['total'] . ' (' . $percent . '%)</div>';
            $html .= '</div>';
        }
        $html .= '</div>';
        $html .= '</div>';
    }
    
    // Distribuição por corporação
    if (!empty($porCorporacao)) {
        $html .= '<div class="report-section">';
        $html .= '<h3><i class="fas fa-shield-alt"></i> Distribuição por Corporação</h3>';
        $html .= '<div class="chart-container">';
        foreach ($porCorporacao as $corp) {
            $percent = round(($corp['total']/$comMilitar)*100, 1);
            $html .= '<div class="chart-bar">';
            $html .= '<div class="chart-label">' . $corp['corporacao'] . '</div>';
            $html .= '<div class="chart-progress">';
            $html .= '<div class="chart-fill" style="width: ' . $percent . '%"></div>';
            $html .= '</div>';
            $html .= '<div class="chart-value">' . $corp['total'] . ' (' . $percent . '%)</div>';
            $html .= '</div>';
        }
        $html .= '</div>';
        $html .= '</div>';
    }
    
    // Últimos registros inseridos
    $html .= '<div class="report-section">';
    $html .= '<h3><i class="fas fa-clock"></i> Últimos Registros Inseridos</h3>';
    $stmt = $pdo->query("
        SELECT a.nome, a.cpf, m.corporacao, m.patente, a.id
        FROM Associados a
        LEFT JOIN Militar m ON a.id = m.associado_id
        ORDER BY a.id DESC
        LIMIT 10
    ");
    $ultimos = $stmt->fetchAll();
    
    if (!empty($ultimos)) {
        $html .= '<div class="table-container">';
        $html .= '<table class="report-table">';
        $html .= '<thead>';
        $html .= '<tr><th>ID</th><th>Nome</th><th>CPF</th><th>Corporação</th><th>Patente</th></tr>';
        $html .= '</thead>';
        $html .= '<tbody>';
        foreach ($ultimos as $reg) {
            $html .= '<tr>';
            $html .= '<td>' . $reg['id'] . '</td>';
            $html .= '<td>' . htmlspecialchars($reg['nome']) . '</td>';
            $html .= '<td>' . formatarCPF($reg['cpf']) . '</td>';
            $html .= '<td>' . ($reg['corporacao'] ?: '-') . '</td>';
            $html .= '<td>' . ($reg['patente'] ?: '-') . '</td>';
            $html .= '</tr>';
        }
        $html .= '</tbody>';
        $html .= '</table>';
        $html .= '</div>';
    }
    $html .= '</div>';
    
    // Informações de backup
    $html .= '<div class="report-section">';
    $html .= '<h3><i class="fas fa-database"></i> Backups Disponíveis</h3>';
    
    $stmt = $pdo->query("SHOW TABLES LIKE '%_backup_%'");
    $backups = $stmt->fetchAll();
    
    if (!empty($backups)) {
        $html .= '<div class="backup-list">';
        foreach ($backups as $backup) {
            $tableName = array_values($backup)[0];
            $html .= '<div class="backup-item">';
            $html .= '<i class="fas fa-archive"></i>';
            $html .= '<span>' . $tableName . '</span>';
            $html .= '</div>';
        }
        $html .= '</div>';
    } else {
        $html .= '<p>Nenhum backup encontrado.</p>';
    }
    
    $html .= '</div>';
    $html .= '</div>'; // report-container
    
    // Adicionar estilos CSS
    $html .= '<style>
        .report-container { font-family: Arial, sans-serif; }
        .report-section { margin-bottom: 30px; padding: 20px; background: #f8f9fa; border-radius: 10px; }
        .report-section h3 { color: #2c3e50; margin-bottom: 20px; display: flex; align-items: center; gap: 10px; }
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; }
        .stat-card { background: white; padding: 20px; border-radius: 8px; text-align: center; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .stat-number { font-size: 2rem; font-weight: bold; color: #667eea; }
        .stat-label { color: #7f8c8d; margin-top: 5px; }
        .stat-percent { color: #27ae60; font-weight: bold; margin-top: 5px; }
        .chart-container { background: white; padding: 20px; border-radius: 8px; }
        .chart-bar { margin-bottom: 15px; }
        .chart-label { font-weight: bold; margin-bottom: 5px; color: #2c3e50; }
        .chart-progress { background: #ecf0f1; height: 25px; border-radius: 12px; position: relative; margin-bottom: 5px; }
        .chart-fill { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); height: 100%; border-radius: 12px; transition: width 0.3s ease; }
        .chart-value { text-align: right; color: #7f8c8d; font-size: 0.9rem; }
        .table-container { background: white; border-radius: 8px; overflow: hidden; }
        .report-table { width: 100%; border-collapse: collapse; }
        .report-table th { background: #667eea; color: white; padding: 12px; text-align: left; }
        .report-table td { padding: 12px; border-bottom: 1px solid #ecf0f1; }
        .report-table tr:hover { background: #f8f9fa; }
        .backup-list { display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); gap: 10px; }
        .backup-item { background: white; padding: 15px; border-radius: 8px; display: flex; align-items: center; gap: 10px; }
        .backup-item i { color: #667eea; }
    </style>';
    
    return $html;
}

function formatarCPF($cpf) {
    if (strlen($cpf) === 11) {
        return substr($cpf, 0, 3) . '.' . substr($cpf, 3, 3) . '.' . substr($cpf, 6, 3) . '-' . substr($cpf, 9, 2);
    }
    return $cpf;
}
?>