<?php
// Importador CSV SIMPLIFICADO - Sem problemas de transação
ini_set('max_execution_time', 0);
ini_set('memory_limit', '512M');
set_time_limit(0);

header('Content-Type: text/plain; charset=utf-8');
header('Cache-Control: no-cache');
header('Connection: keep-alive');

while (ob_get_level()) {
    ob_end_clean();
}

function sendUpdate($data) {
    echo json_encode($data) . "\n";
    if (connection_status() === CONNECTION_NORMAL) {
        flush();
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
    }
}

function sendLog($message, $level = 'info') {
    sendUpdate([
        'type' => 'log',
        'message' => $message,
        'level' => $level,
        'timestamp' => date('H:i:s')
    ]);
}

function checkConnection() {
    if (connection_status() !== CONNECTION_NORMAL) {
        throw new Exception('Conexão perdida');
    }
}

try {
    sendLog('🚀 Iniciando importação CSV SIMPLIFICADA...', 'info');
    
    // ⚠️ ALERTA IMPORTANTE SOBRE BACKUP
    sendLog('', 'info');
    sendLog('🛡️ =================== ATENÇÃO =================== 🛡️', 'warning');
    sendLog('⚠️  CERTIFIQUE-SE DE TER FEITO BACKUP DO BANCO!  ⚠️', 'warning');
    sendLog('📝 Este processo irá INSERIR e ATUALIZAR dados', 'warning');
    sendLog('💡 Recomendamos backup via phpMyAdmin ou mysqldump', 'warning');
    sendLog('🛡️ ============================================= 🛡️', 'warning');
    sendLog('', 'info');
    
    // Configurar arquivo CSV
    $csvFile = null;
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $csvFile = $_FILES['csv_file']['tmp_name'];
        sendLog('📁 Usando arquivo enviado via POST');
    } else {
        $possibleFiles = ['../db.csv', '../dados.csv', '../uploads/dados.csv', './db.csv', './dados.csv'];
        foreach ($possibleFiles as $file) {
            if (file_exists($file)) {
                $csvFile = $file;
                sendLog("📁 Usando arquivo: $file");
                break;
            }
        }
        
        if (!$csvFile) {
            throw new Exception('Arquivo CSV não encontrado');
        }
    }
    
    // Parâmetros
    $updateExisting = isset($_POST['update_existing']) ? $_POST['update_existing'] === 'true' : true;
    $batchSize = intval($_POST['batch_size'] ?? 50);
    
    sendLog("⚙️ Configurações:");
    sendLog("   • Lote: {$batchSize} registros");
    sendLog("   • Atualizar existentes: " . ($updateExisting ? 'SIM' : 'NÃO'));
    
    // Conectar ao banco
    $configFile = '../config/database.json';
    if (!file_exists($configFile)) {
        throw new Exception('Configuração do banco não encontrada');
    }
    
    $config = json_decode(file_get_contents($configFile), true);
    if (!$config) {
        throw new Exception('Erro ao ler configuração do banco');
    }
    
    sendLog('🔌 Conectando ao banco de dados...');
    
    $pdo = new PDO(
        "mysql:host={$config['host']};dbname={$config['dbname']};charset=utf8mb4",
        $config['username'],
        $config['password'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
        ]
    );
    
    // Configurações MySQL básicas
    $pdo->exec("SET SESSION sql_mode = 'ALLOW_INVALID_DATES'");
    
    sendLog('✅ Conexão estabelecida!', 'success');
    
    // Verificar dados existentes
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM Associados");
    $totalExistente = $stmt->fetch()['total'];
    sendLog("📊 Associados no banco: {$totalExistente}");
    
    // Abrir e analisar arquivo
    $handle = fopen($csvFile, 'r');
    if (!$handle) {
        throw new Exception('Erro ao abrir arquivo CSV');
    }
    
    sendLog('🔍 Analisando arquivo...');
    
    // Processar cabeçalho
    $delimiter = ';';
    $header = fgetcsv($handle, 0, $delimiter);
    if (!$header) {
        throw new Exception('Erro ao ler cabeçalho');
    }
    
    $header = array_map('trim', $header);
    
    // Remover colunas vazias no final
    while (end($header) === '') {
        array_pop($header);
        sendLog('🗑️ Removida coluna vazia');
    }
    
    $expectedColumns = count($header);
    sendLog("📋 Colunas: {$expectedColumns}");
    
    // Contar registros
    $totalRegistros = 0;
    while (fgetcsv($handle, 0, $delimiter) !== false) {
        $totalRegistros++;
        if ($totalRegistros % 1000 === 0) {
            sendLog("📊 Contados: {$totalRegistros}...");
            checkConnection();
        }
    }
    sendLog("📊 Total: {$totalRegistros} registros", 'success');
    
    // Voltar ao início
    rewind($handle);
    fgetcsv($handle, 0, $delimiter); // Pular cabeçalho
    
    // Mapeamento simplificado
    $mapeamento = [
        'nome_ass' => 'nome',
        'data_nasc' => 'nasc', 
        'sexo' => 'sexo',
        'rg' => 'rg',
        'cpf' => 'cpf',
        'situação' => 'situacao',
        'escolaridade' => 'escolaridade',
        'est_civil' => 'estadoCivil',
        'email' => 'email',
        'fone_celular' => 'telefone',
        'cep' => 'cep',
        'endereco' => 'endereco',
        'bairro' => 'bairro',
        'cidade' => 'cidade',
        'corporação' => 'corporacao',
        'patente' => 'patente',
        'categoria' => 'categoria',
        'lotação' => 'lotacao',
        'vinculo' => 'vinculoServidor',
        'facebook' => 'facebook',
        'instagram' => 'instagram',
        'twitter' => 'twitter'
    ];
    
    // Preparar statements
    $stmtCheck = $pdo->prepare("SELECT id FROM Associados WHERE cpf = ? LIMIT 1");
    
    $stmtInsert = $pdo->prepare("
        INSERT INTO Associados (nome, nasc, sexo, rg, cpf, email, situacao, escolaridade, estadoCivil, telefone) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmtUpdate = $pdo->prepare("
        UPDATE Associados SET nome=?, nasc=?, sexo=?, rg=?, email=?, situacao=?, escolaridade=?, estadoCivil=?, telefone=? 
        WHERE id=?
    ");
    
    $stmtEndereco = $pdo->prepare("
        INSERT INTO Endereco (associado_id, cep, endereco, bairro, cidade) 
        VALUES (?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE cep=VALUES(cep), endereco=VALUES(endereco), bairro=VALUES(bairro), cidade=VALUES(cidade)
    ");
    
    $stmtMilitar = $pdo->prepare("
        INSERT INTO Militar (associado_id, corporacao, patente, categoria, lotacao) 
        VALUES (?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE corporacao=VALUES(corporacao), patente=VALUES(patente), categoria=VALUES(categoria), lotacao=VALUES(lotacao)
    ");
    
    $stmtFinanceiro = $pdo->prepare("
        INSERT INTO Financeiro (associado_id, vinculoServidor) 
        VALUES (?, ?)
        ON DUPLICATE KEY UPDATE vinculoServidor=VALUES(vinculoServidor)
    ");
    
    $stmtRede = $pdo->prepare("
        INSERT INTO Redes_sociais (associado_id, nome_rede, usuario) 
        VALUES (?, ?, ?)
        ON DUPLICATE KEY UPDATE usuario=VALUES(usuario)
    ");
    
    sendLog('📝 Statements preparados');
    
    // Contadores
    $stats = [
        'total' => $totalRegistros,
        'processed' => 0,
        'inserted' => 0,
        'updated' => 0,
        'errors' => 0,
        'skipped' => 0
    ];
    
    sendLog('⚡ Iniciando processamento...');
    
    $processed = 0;
    $batchCount = 0;
    $startTime = time();
    
    // Processar linha por linha (SEM transações complexas)
    while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
        try {
            $processed++;
            $batchCount++;
            
            checkConnection();
            
            if (count($row) !== $expectedColumns) {
                $stats['errors']++;
                continue;
            }
            
            // Combinar e mapear dados
            $dadosOriginais = array_combine($header, $row);
            $dadosOriginais = array_map('trim', $dadosOriginais);
            
            $data = [];
            foreach ($mapeamento as $original => $novo) {
                $data[$novo] = $dadosOriginais[$original] ?? '';
            }
            
            // Normalizar
            $data = normalizarDados($data);
            
            // Validar
            if (empty($data['cpf']) || strlen($data['cpf']) !== 11 || empty($data['nome'])) {
                $stats['skipped']++;
                continue;
            }
            
            // Verificar se existe
            $stmtCheck->execute([$data['cpf']]);
            $associadoId = $stmtCheck->fetchColumn();
            
            if ($associadoId && $updateExisting) {
                // Atualizar
                $stmtUpdate->execute([
                    $data['nome'],
                    formatarData($data['nasc']),
                    $data['sexo'] ?: null,
                    $data['rg'] ?: null,
                    $data['email'] ?: null,
                    $data['situacao'] ?: null,
                    $data['escolaridade'] ?: null,
                    $data['estadoCivil'] ?: null,
                    $data['telefone'] ?: null,
                    $associadoId
                ]);
                $stats['updated']++;
                
            } elseif (!$associadoId) {
                // Inserir novo
                $stmtInsert->execute([
                    $data['nome'],
                    formatarData($data['nasc']),
                    $data['sexo'] ?: null,
                    $data['rg'] ?: null,
                    $data['cpf'],
                    $data['email'] ?: null,
                    $data['situacao'] ?: null,
                    $data['escolaridade'] ?: null,
                    $data['estadoCivil'] ?: null,
                    $data['telefone'] ?: null
                ]);
                
                $associadoId = $pdo->lastInsertId();
                $stats['inserted']++;
            } else {
                $stats['skipped']++;
            }
            
            // Inserir dados relacionados
            if ($associadoId) {
                // Endereço
                if (!empty($data['cep']) || !empty($data['endereco'])) {
                    $stmtEndereco->execute([
                        $associadoId,
                        $data['cep'] ?: null,
                        $data['endereco'] ?: null,
                        $data['bairro'] ?: null,
                        $data['cidade'] ?: null
                    ]);
                }
                
                // Militar
                if (!empty($data['corporacao']) || !empty($data['patente'])) {
                    $stmtMilitar->execute([
                        $associadoId,
                        $data['corporacao'] ?: null,
                        $data['patente'] ?: null,
                        $data['categoria'] ?: null,
                        $data['lotacao'] ?: null
                    ]);
                }
                
                // Financeiro
                if (!empty($data['vinculoServidor'])) {
                    $stmtFinanceiro->execute([
                        $associadoId,
                        $data['vinculoServidor']
                    ]);
                }
                
                // Redes sociais (apenas se tiver dados)
                $redes = ['facebook', 'instagram', 'twitter'];
                foreach ($redes as $rede) {
                    if (!empty($data[$rede])) {
                        $stmtRede->execute([
                            $associadoId,
                            $rede,
                            $data[$rede]
                        ]);
                    }
                }
            }
            
            $stats['processed'] = $processed;
            
            // Progress update a cada lote
            if ($batchCount >= $batchSize) {
                sendUpdate([
                    'type' => 'progress',
                    'total' => $stats['total'],
                    'processed' => $stats['processed'],
                    'inserted' => $stats['inserted'],
                    'updated' => $stats['updated'],
                    'errors' => $stats['errors'],
                    'skipped' => $stats['skipped']
                ]);
                
                $elapsed = time() - $startTime;
                $speed = $elapsed > 0 ? round($batchCount / $elapsed, 2) : 0;
                $remaining = $totalRegistros - $processed;
                $eta = $speed > 0 ? round($remaining / $speed / 60, 1) : 0;
                $percent = round(($processed / $totalRegistros) * 100, 1);
                
                sendLog("⚡ {$percent}% | {$speed} reg/s | ETA: {$eta}min | I:{$stats['inserted']} U:{$stats['updated']} E:{$stats['errors']} S:{$stats['skipped']}");
                
                $batchCount = 0;
                $startTime = time();
            }
            
        } catch (Exception $e) {
            $stats['errors']++;
            sendLog("❌ Erro linha {$processed}: " . $e->getMessage(), 'error');
        }
    }
    
    fclose($handle);
    
    // Resultado final
    sendUpdate([
        'type' => 'complete',
        'total' => $stats['total'],
        'processed' => $stats['processed'],
        'inserted' => $stats['inserted'],
        'updated' => $stats['updated'],
        'errors' => $stats['errors'],
        'skipped' => $stats['skipped']
    ]);
    
    sendLog('', 'info');
    sendLog('🎉 ================= IMPORTAÇÃO CONCLUÍDA ================= 🎉', 'success');
    sendLog("📊 RESUMO:", 'success');
    sendLog("   ✅ {$stats['inserted']} INSERIDOS", 'success');
    sendLog("   🔄 {$stats['updated']} ATUALIZADOS", 'success');
    sendLog("   ❌ {$stats['errors']} erros", $stats['errors'] > 0 ? 'warning' : 'info');
    sendLog("   ⏭️ {$stats['skipped']} ignorados", $stats['skipped'] > 0 ? 'warning' : 'info');
    
    $totalProcessed = $stats['inserted'] + $stats['updated'];
    $successRate = round(($totalProcessed / $stats['total']) * 100, 1);
    sendLog("🏆 TOTAL: {$totalProcessed} registros | Taxa: {$successRate}%", 'success');
    
} catch (Exception $e) {
    sendUpdate([
        'type' => 'error',
        'message' => $e->getMessage()
    ]);
    sendLog('💥 ERRO: ' . $e->getMessage(), 'error');
}

// === FUNÇÕES AUXILIARES ===

function normalizarDados($data) {
    // Sexo
    if (isset($data['sexo'])) {
        $sexo = strtolower(trim($data['sexo']));
        if (in_array($sexo, ['masculino', 'masc', 'm'])) {
            $data['sexo'] = 'M';
        } elseif (in_array($sexo, ['feminino', 'fem', 'f'])) {
            $data['sexo'] = 'F';
        }
    }
    
    // CPF
    if (isset($data['cpf'])) {
        $data['cpf'] = preg_replace('/[^0-9]/', '', $data['cpf']);
        if (strlen($data['cpf']) !== 11) {
            $data['cpf'] = '';
        }
    }
    
    // RG
    if (isset($data['rg'])) {
        $data['rg'] = preg_replace('/[^0-9]/', '', $data['rg']);
    }
    
    // Limpar vazios
    foreach ($data as $campo => $valor) {
        if (in_array($valor, ['0000-00-00', 'NULL', 'null', '', '0'])) {
            $data[$campo] = '';
        }
    }
    
    return $data;
}

function formatarData($data) {
    if (empty($data)) return null;
    
    $data = trim($data);
    
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        return ($data === '0000-00-00') ? null : $data;
    }
    
    if (preg_match('/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/', $data, $matches)) {
        return sprintf('%04d-%02d-%02d', $matches[3], $matches[2], $matches[1]);
    }
    
    return null;
}
?>