<?php
/**
 * Arquivo de configuração principal
 * includes/config.php
 */

// Configurações gerais
define('APP_NAME', 'Sistema de Importação WWASSE');
define('APP_VERSION', '1.0.0');
define('APP_TIMEZONE', 'America/Sao_Paulo');

// Configurar timezone
date_default_timezone_set(APP_TIMEZONE);

// Configurações de erro
error_reporting(E_ALL);
ini_set('display_errors', 0); // Não mostrar erros em produção
ini_set('log_errors', 1);
ini_set('error_log', '../logs/php_errors.log');

// Configurações de upload
ini_set('upload_max_filesize', '50M');
ini_set('post_max_size', '50M');
ini_set('max_execution_time', 300); // 5 minutos
ini_set('memory_limit', '256M');

// Função para carregar configuração do banco
function getDbConfig() {
    $configFile = __DIR__ . '/../config/database.json';
    
    if (!file_exists($configFile)) {
        return null;
    }
    
    $config = json_decode(file_get_contents($configFile), true);
    return $config ?: null;
}

// Função para conectar ao banco
function getDbConnection() {
    $config = getDbConfig();
    
    if (!$config) {
        throw new Exception('Configuração do banco não encontrada');
    }
    
    try {
        $pdo = new PDO(
            "mysql:host={$config['host']};dbname={$config['dbname']};charset=utf8mb4",
            $config['username'],
            $config['password'],
            [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ]
        );
        
        return $pdo;
    } catch (PDOException $e) {
        throw new Exception('Erro de conexão: ' . $e->getMessage());
    }
}

// Função para log
function writeLog($message, $level = 'INFO') {
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logFile = $logDir . '/app.log';
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message" . PHP_EOL;
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

// Função para validar CSRF (se necessário)
function validateCSRF($token) {
    session_start();
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Função para gerar CSRF token
function generateCSRF() {
    session_start();
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Função para sanitizar dados
function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return trim(strip_tags($data));
}

// Função para validar email
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Função para validar CPF
function isValidCPF($cpf) {
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    
    if (strlen($cpf) != 11 || preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    
    for ($t = 9; $t < 11; $t++) {
        $d = 0;
        for ($c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) {
            return false;
        }
    }
    
    return true;
}

// Autoload simples (se necessário)
spl_autoload_register(function ($class) {
    $file = __DIR__ . '/classes/' . $class . '.php';
    if (file_exists($file)) {
        require_once $file;
    }
});

// Verificar se diretórios necessários existem
$requiredDirs = [
    __DIR__ . '/../config',
    __DIR__ . '/../logs',
    __DIR__ . '/../uploads'
];

foreach ($requiredDirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Configurar headers de segurança
function setSecurityHeaders() {
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    header('Referrer-Policy: strict-origin-when-cross-origin');
}

// Função para limpar arquivos temporários antigos
function cleanupTempFiles() {
    $uploadDir = __DIR__ . '/../uploads';
    if (!is_dir($uploadDir)) return;
    
    $files = glob($uploadDir . '/*');
    $now = time();
    
    foreach ($files as $file) {
        if (is_file($file) && ($now - filemtime($file)) > 3600) { // 1 hora
            unlink($file);
        }
    }
}

// Executar limpeza de arquivos temporários ocasionalmente
if (rand(1, 100) === 1) {
    cleanupTempFiles();
}
?>