// Variáveis globais
let selectedFile = null;
let importProgress = {
    total: 0,
    processed: 0,
    inserted: 0,
    updated: 0,
    errors: 0
};

// Event Listeners
document.addEventListener('DOMContentLoaded', function() {
    initializeFileUpload();
    initializeFormHandlers();
    initializeLargeFileMode();
});

// Inicializar modo arquivo grande
function initializeLargeFileMode() {
    const largeFileCheckbox = document.getElementById('large-file-mode');
    const advancedOptions = document.getElementById('advanced-options');
    
    largeFileCheckbox.addEventListener('change', function() {
        if (this.checked) {
            advancedOptions.style.display = 'block';
        } else {
            advancedOptions.style.display = 'none';
        }
    });
}

// Inicializar upload de arquivo
function initializeFileUpload() {
    const uploadArea = document.getElementById('upload-area');
    const fileInput = document.getElementById('csv-file');
    
    // Drag and drop
    uploadArea.addEventListener('dragover', function(e) {
        e.preventDefault();
        uploadArea.classList.add('dragover');
    });
    
    uploadArea.addEventListener('dragleave', function(e) {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
    });
    
    uploadArea.addEventListener('drop', function(e) {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            handleFileSelect(files[0]);
        }
    });
    
    // Click para selecionar
    uploadArea.addEventListener('click', function() {
        fileInput.click();
    });
    
    // Mudança de arquivo
    fileInput.addEventListener('change', function(e) {
        if (e.target.files.length > 0) {
            handleFileSelect(e.target.files[0]);
        }
    });
}

// Inicializar handlers de formulário
function initializeFormHandlers() {
    const dbConfigForm = document.getElementById('db-config-form');
    dbConfigForm.addEventListener('submit', function(e) {
        e.preventDefault();
        saveConfiguration();
    });
}

// Manipular seleção de arquivo
function handleFileSelect(file) {
    if (!file.name.toLowerCase().endsWith('.csv')) {
        showAlert('Por favor, selecione um arquivo CSV válido.', 'error');
        return;
    }
    
    selectedFile = file;
    showFileInfo(file);
    previewFile(file);
    checkFileSize(file);
    updateImportButton();
}

// Verificar tamanho do arquivo e dar dicas
function checkFileSize(file) {
    const fileInfo = document.getElementById('file-info');
    const size = file.size;
    const sizeMB = (size / 1024 / 1024).toFixed(2);
    
    // Remover indicadores anteriores
    const existingIndicators = fileInfo.querySelectorAll('.file-size-warning, .file-size-ok, .performance-tips');
    existingIndicators.forEach(el => el.remove());
    
    if (size > 10 * 1024 * 1024) { // > 10MB
        // Arquivo muito grande
        const warning = document.createElement('div');
        warning.className = 'file-size-warning';
        warning.innerHTML = `
            <i class="fas fa-exclamation-triangle"></i>
            <span>Arquivo grande detectado (${sizeMB} MB). Modo otimizado será ativado automaticamente.</span>
        `;
        fileInfo.appendChild(warning);
        
        // Ativar modo arquivo grande automaticamente
        document.getElementById('large-file-mode').checked = true;
        document.getElementById('advanced-options').style.display = 'block';
        
        // Mostrar dicas de performance
        showPerformanceTips(sizeMB);
        
    } else if (size > 5 * 1024 * 1024) { // 5-10MB
        // Arquivo médio
        const warning = document.createElement('div');
        warning.className = 'file-size-warning';
        warning.innerHTML = `
            <i class="fas fa-info-circle"></i>
            <span>Arquivo médio (${sizeMB} MB). Considere usar o modo otimizado se houver problemas.</span>
        `;
        fileInfo.appendChild(warning);
        
    } else {
        // Arquivo pequeno
        const ok = document.createElement('div');
        ok.className = 'file-size-ok';
        ok.innerHTML = `
            <i class="fas fa-check-circle"></i>
            <span>Tamanho adequado (${sizeMB} MB). Importação rápida esperada.</span>
        `;
        fileInfo.appendChild(ok);
    }
}

// Mostrar dicas de performance
function showPerformanceTips(sizeMB) {
    const fileInfo = document.getElementById('file-info');
    
    const tips = document.createElement('div');
    tips.className = 'performance-tips';
    
    let batchRecommendation = '50';
    if (sizeMB > 50) batchRecommendation = '25';
    if (sizeMB > 100) batchRecommendation = '25';
    
    tips.innerHTML = `
        <h5><i class="fas fa-lightbulb"></i> Dicas para Arquivos Grandes</h5>
        <ul>
            <li>Tamanho de lote recomendado: <strong>${batchRecommendation} registros</strong></li>
            <li>A importação pode levar vários minutos</li>
            <li>Mantenha a aba aberta durante o processo</li>
            <li>O backup é criado automaticamente para segurança</li>
            <li>Você pode retomar se a importação for interrompida</li>
        </ul>
    `;
    
    fileInfo.appendChild(tips);
    
    // Ajustar tamanho do lote automaticamente
    document.getElementById('batch-size').value = batchRecommendation;
}

// Mostrar informações do arquivo
function showFileInfo(file) {
    const fileInfo = document.getElementById('file-info');
    const fileName = document.getElementById('file-name');
    const fileSize = document.getElementById('file-size');
    
    fileName.textContent = file.name;
    fileSize.textContent = formatFileSize(file.size);
    fileInfo.style.display = 'block';
    
    document.getElementById('upload-area').style.display = 'none';
}

// Preview do arquivo
function previewFile(file) {
    const reader = new FileReader();
    reader.onload = function(e) {
        const text = e.target.result;
        const lines = text.split('\n').slice(0, 5); // Primeiras 5 linhas
        
        const preview = document.getElementById('file-preview');
        preview.innerHTML = '<strong>Preview (primeiras 5 linhas):</strong><br><br>' + 
                           lines.map(line => escapeHtml(line)).join('<br>');
    };
    reader.readAsText(file);
}

// Remover arquivo
function removeFile() {
    selectedFile = null;
    document.getElementById('file-info').style.display = 'none';
    document.getElementById('upload-area').style.display = 'block';
    document.getElementById('csv-file').value = '';
    updateImportButton();
}

// Formatar tamanho do arquivo
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Verificar status do banco de dados
async function checkDatabaseStatus() {
    try {
        const response = await fetch('api/check_status.php');
        const data = await response.json();
        
        updateStatusIndicator('db-indicator', data.database.status, data.database.message);
        updateStatusIndicator('tables-indicator', data.tables.status, data.tables.message);
        
        updateImportButton();
    } catch (error) {
        updateStatusIndicator('db-indicator', 'error', 'Erro ao verificar conexão');
        updateStatusIndicator('tables-indicator', 'error', 'Erro ao verificar tabelas');
    }
}

// Atualizar indicador de status
function updateStatusIndicator(elementId, status, message) {
    const indicator = document.getElementById(elementId);
    
    let icon, className;
    switch (status) {
        case 'success':
            icon = '<i class="fas fa-check"></i>';
            className = 'status-success';
            break;
        case 'error':
            icon = '<i class="fas fa-times"></i>';
            className = 'status-error';
            break;
        case 'warning':
            icon = '<i class="fas fa-exclamation-triangle"></i>';
            className = 'status-warning';
            break;
        default:
            icon = '<i class="fas fa-spinner fa-spin"></i>';
            className = '';
    }
    
    indicator.innerHTML = icon;
    indicator.className = 'status-indicator ' + className;
    indicator.title = message;
}

// Carregar configuração salva
async function loadConfig() {
    try {
        const response = await fetch('api/get_config.php');
        const data = await response.json();
        
        if (data.success) {
            document.getElementById('host').value = data.config.host || 'localhost';
            document.getElementById('dbname').value = data.config.dbname || 'wwasse_cadastro';
            document.getElementById('username').value = data.config.username || '';
            // Não carregar senha por segurança
        }
    } catch (error) {
        console.log('Nenhuma configuração salva encontrada');
    }
}

// Salvar configuração
async function saveConfiguration() {
    const formData = new FormData();
    formData.append('host', document.getElementById('host').value);
    formData.append('dbname', document.getElementById('dbname').value);
    formData.append('username', document.getElementById('username').value);
    formData.append('password', document.getElementById('password').value);
    
    try {
        showLoading(true);
        const response = await fetch('api/save_config.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showAlert('Configuração salva com sucesso!', 'success');
            toggleConfig(); // Fechar formulário
            checkDatabaseStatus(); // Verificar nova configuração
        } else {
            showAlert('Erro ao salvar configuração: ' + data.message, 'error');
        }
    } catch (error) {
        showAlert('Erro de comunicação: ' + error.message, 'error');
    } finally {
        showLoading(false);
    }
}

// Toggle configuração
function toggleConfig() {
    const configForm = document.getElementById('config-form');
    const toggleBtn = document.querySelector('.btn-toggle');
    
    if (configForm.style.display === 'none') {
        configForm.style.display = 'block';
        toggleBtn.style.transform = 'rotate(180deg)';
    } else {
        configForm.style.display = 'none';
        toggleBtn.style.transform = 'rotate(0deg)';
    }
}

// Atualizar botão de importação
function updateImportButton() {
    const startButton = document.getElementById('start-import');
    const dbStatus = document.getElementById('db-indicator').classList.contains('status-success');
    const tablesStatus = document.getElementById('tables-indicator').classList.contains('status-success');
    
    if (selectedFile && dbStatus && tablesStatus) {
        startButton.disabled = false;
        startButton.innerHTML = '<i class="fas fa-rocket"></i> Iniciar Importação';
    } else {
        startButton.disabled = true;
        if (!selectedFile) {
            startButton.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Selecione um arquivo CSV';
        } else if (!dbStatus || !tablesStatus) {
            startButton.innerHTML = '<i class="fas fa-database"></i> Configure o banco de dados';
        }
    }
}

// Iniciar importação
async function startImport() {
    if (!selectedFile) {
        showAlert('Selecione um arquivo CSV primeiro.', 'error');
        return;
    }
    
    // Detectar se é arquivo grande
    const isLargeFile = document.getElementById('large-file-mode').checked || selectedFile.size > 5 * 1024 * 1024; // > 5MB
    
    if (isLargeFile) {
        showAlert('Modo arquivo grande ativado. Processo otimizado iniciado.', 'info');
    }
    
    // Mostrar card de progresso
    document.getElementById('progress-card').style.display = 'block';
    document.getElementById('results-card').style.display = 'none';
    
    // Resetar progresso
    resetProgress();
    
    // Preparar FormData
    const formData = new FormData();
    formData.append('csv_file', selectedFile);
    formData.append('create_backup', document.getElementById('create-backup').checked);
    formData.append('update_existing', document.getElementById('update-existing').checked);
    
    if (isLargeFile) {
        formData.append('batch_size', document.getElementById('batch-size').value);
        formData.append('start_from', document.getElementById('start-from').value);
    }
    
    try {
        // Desabilitar botão
        const startButton = document.getElementById('start-import');
        startButton.disabled = true;
        startButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Importando...';
        
        // Escolher API baseada no tamanho
        const apiUrl = isLargeFile ? 'api/import_large.php' : 'api/import.php';
        
        addLogEntry(`Iniciando importação via ${isLargeFile ? 'API otimizada' : 'API padrão'}...`, 'info');
        
        // Iniciar importação
        const response = await fetch(apiUrl, {
            method: 'POST',
            body: formData
        });
        
        if (!response.ok) {
            throw new Error('Erro na comunicação com o servidor');
        }
        
        // Ler stream de dados
        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        
        while (true) {
            const { done, value } = await reader.read();
            
            if (done) break;
            
            const chunk = decoder.decode(value);
            const lines = chunk.split('\n');
            
            for (const line of lines) {
                if (line.trim()) {
                    try {
                        const data = JSON.parse(line);
                        handleImportProgress(data);
                    } catch (e) {
                        // Linha não é JSON válido, pode ser log
                        addLogEntry(line, 'info');
                    }
                }
            }
        }
        
    } catch (error) {
        showAlert('Erro durante a importação: ' + error.message, 'error');
        addLogEntry('Erro: ' + error.message, 'error');
        
        // Se for arquivo grande e falhou, sugerir retomar
        if (isLargeFile) {
            showResumeOption();
        }
    } finally {
        // Reabilitar botão
        updateImportButton();
    }
}

// Manipular progresso da importação
function handleImportProgress(data) {
    switch (data.type) {
        case 'progress':
            updateProgress(data);
            break;
        case 'log':
            addLogEntry(data.message, data.level);
            break;
        case 'complete':
            completeImport(data);
            break;
        case 'error':
            addLogEntry(data.message, 'error');
            break;
    }
}

// Atualizar progresso
function updateProgress(data) {
    importProgress = { ...importProgress, ...data };
    
    const percentage = importProgress.total > 0 ? 
        Math.round((importProgress.processed / importProgress.total) * 100) : 0;
    
    document.getElementById('progress-fill').style.width = percentage + '%';
    document.getElementById('progress-text').textContent = percentage + '%';
    
    document.getElementById('stat-processed').textContent = importProgress.processed || 0;
    document.getElementById('stat-inserted').textContent = importProgress.inserted || 0;
    document.getElementById('stat-updated').textContent = importProgress.updated || 0;
    document.getElementById('stat-errors').textContent = importProgress.errors || 0;
    
    // Adicionar estatística de ignorados se disponível
    if (data.skipped !== undefined) {
        // Criar elemento para skipped se não existir
        if (!document.getElementById('stat-skipped')) {
            const statsContainer = document.querySelector('.progress-stats');
            const skippedStat = document.createElement('div');
            skippedStat.className = 'stat-item';
            skippedStat.innerHTML = `
                <span class="stat-label">Ignorados:</span>
                <span class="stat-value" id="stat-skipped">0</span>
            `;
            statsContainer.appendChild(skippedStat);
        }
        document.getElementById('stat-skipped').textContent = importProgress.skipped || 0;
    }
}

// Mostrar opção de retomar importação
function showResumeOption() {
    const resumeHtml = `
        <div class="resume-option" style="background: #fff3cd; padding: 15px; border-radius: 10px; margin-top: 15px;">
            <h4><i class="fas fa-exclamation-triangle"></i> Importação Interrompida</h4>
            <p>A importação foi interrompida. Você pode retomar de onde parou.</p>
            <button class="btn btn-warning" onclick="resumeImport()">
                <i class="fas fa-play"></i> Retomar Importação
            </button>
        </div>
    `;
    
    const progressCard = document.getElementById('progress-card');
    progressCard.innerHTML += resumeHtml;
}

// Retomar importação
async function resumeImport() {
    const processed = importProgress.processed || 0;
    document.getElementById('start-from').value = processed;
    document.getElementById('large-file-mode').checked = true;
    document.getElementById('advanced-options').style.display = 'block';
    
    showAlert(`Retomando importação da linha ${processed + 1}...`, 'info');
    await startImport();
}

// Adicionar entrada de log
function addLogEntry(message, level = 'info') {
    const logContainer = document.getElementById('import-log');
    const entry = document.createElement('div');
    entry.className = 'log-entry log-' + level;
    
    const timestamp = new Date().toLocaleTimeString();
    entry.innerHTML = `[${timestamp}] ${escapeHtml(message)}`;
    
    logContainer.appendChild(entry);
    logContainer.scrollTop = logContainer.scrollHeight;
}

// Completar importação
function completeImport(data) {
    // Atualizar progresso final
    updateProgress(data);
    
    // Mostrar resultados
    setTimeout(() => {
        document.getElementById('results-card').style.display = 'block';
        
        const summary = document.getElementById('results-summary');
        summary.innerHTML = `
            <h3>✅ Importação Concluída!</h3>
            <p>
                <strong>${data.processed}</strong> registros processados • 
                <strong>${data.inserted}</strong> inseridos • 
                <strong>${data.updated}</strong> atualizados • 
                <strong>${data.errors}</strong> erros
            </p>
        `;
        
        addLogEntry('Importação finalizada com sucesso!', 'success');
    }, 1000);
}

// Resetar progresso
function resetProgress() {
    importProgress = { total: 0, processed: 0, inserted: 0, updated: 0, errors: 0 };
    updateProgress(importProgress);
    document.getElementById('import-log').innerHTML = '';
}

// Mostrar relatório
async function showReport() {
    try {
        const response = await fetch('api/report.php');
        const data = await response.json();
        
        if (data.success) {
            document.getElementById('report-content').innerHTML = data.report;
            document.getElementById('report-modal').style.display = 'block';
        } else {
            showAlert('Erro ao gerar relatório: ' + data.message, 'error');
        }
    } catch (error) {
        showAlert('Erro ao carregar relatório: ' + error.message, 'error');
    }
}

// Fechar modal
function closeModal() {
    document.getElementById('report-modal').style.display = 'none';
}

// Resetar importação
function resetImport() {
    document.getElementById('progress-card').style.display = 'none';
    document.getElementById('results-card').style.display = 'none';
    removeFile();
    resetProgress();
}

// Mostrar alerta
function showAlert(message, type = 'info') {
    // Criar elemento de alerta
    const alert = document.createElement('div');
    alert.className = 'alert alert-' + type;
    alert.innerHTML = `
        <div class="alert-content">
            <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'times' : 'info'}"></i>
            <span>${escapeHtml(message)}</span>
            <button class="alert-close" onclick="this.parentElement.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    // Adicionar estilos se não existirem
    if (!document.getElementById('alert-styles')) {
        const style = document.createElement('style');
        style.id = 'alert-styles';
        style.textContent = `
            .alert {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 1001;
                max-width: 400px;
                border-radius: 10px;
                box-shadow: 0 5px 15px rgba(0,0,0,0.2);
                animation: slideInRight 0.3s ease;
            }
            .alert-success { background: #d4edda; color: #155724; border-left: 4px solid #28a745; }
            .alert-error { background: #f8d7da; color: #721c24; border-left: 4px solid #dc3545; }
            .alert-info { background: #d1ecf1; color: #0c5460; border-left: 4px solid #17a2b8; }
            .alert-content {
                display: flex;
                align-items: center;
                gap: 10px;
                padding: 15px;
            }
            .alert-content i { font-size: 1.2rem; }
            .alert-content span { flex: 1; }
            .alert-close {
                background: none;
                border: none;
                cursor: pointer;
                padding: 5px;
                border-radius: 50%;
                transition: background 0.3s ease;
            }
            .alert-close:hover { background: rgba(0,0,0,0.1); }
            @keyframes slideInRight {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Adicionar ao DOM
    document.body.appendChild(alert);
    
    // Remover automaticamente após 5 segundos
    setTimeout(() => {
        if (alert.parentElement) {
            alert.remove();
        }
    }, 5000);
}

// Mostrar/ocultar loading
function showLoading(show) {
    const elements = document.querySelectorAll('.btn, .card');
    elements.forEach(el => {
        if (show) {
            el.classList.add('loading');
        } else {
            el.classList.remove('loading');
        }
    });
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Fechar modal ao clicar fora
window.addEventListener('click', function(e) {
    const modal = document.getElementById('report-modal');
    if (e.target === modal) {
        closeModal();
    }
});

// Tratamento de erros globais
window.addEventListener('error', function(e) {
    console.error('Erro JavaScript:', e.error);
    showAlert('Erro inesperado: ' + e.message, 'error');
});

// Tratamento de promessas rejeitadas
window.addEventListener('unhandledrejection', function(e) {
    console.error('Promise rejeitada:', e.reason);
    showAlert('Erro de comunicação: ' + e.reason, 'error');
});