<?php
// Configurações otimizadas para arquivos grandes
ini_set('max_execution_time', 0);
ini_set('memory_limit', '512M');
set_time_limit(0);

// Headers para streaming
header('Content-Type: text/plain; charset=utf-8');
header('Cache-Control: no-cache');
header('Connection: keep-alive');

// Desabilitar output buffering
while (ob_get_level()) {
    ob_end_clean();
}

// Função para enviar dados em tempo real
function sendUpdate($data) {
    echo json_encode($data) . "\n";
    if (connection_status() === CONNECTION_NORMAL) {
        flush();
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
    }
}

// Função para log
function sendLog($message, $level = 'info') {
    sendUpdate([
        'type' => 'log',
        'message' => $message,
        'level' => $level,
        'timestamp' => date('H:i:s')
    ]);
}

// Verificar conexão ativa
function checkConnection() {
    if (connection_status() !== CONNECTION_NORMAL) {
        throw new Exception('Conexão perdida');
    }
}

try {
    sendLog('Iniciando importação otimizada com debug melhorado...', 'info');
    
    // Verificar método e arquivo
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método não permitido');
    }
    
    if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Arquivo CSV não enviado corretamente');
    }
    
    // Parâmetros
    $createBackup = isset($_POST['create_backup']) && $_POST['create_backup'] === 'true';
    $updateExisting = isset($_POST['update_existing']) && $_POST['update_existing'] === 'true';
    $batchSize = intval($_POST['batch_size'] ?? 10); // REDUZIDO para debug
    $startFrom = intval($_POST['start_from'] ?? 0);
    
    sendLog("Configurações: Lote={$batchSize}, Início={$startFrom}");
    
    // Carregar configuração do banco
    $configFile = '../config/database.json';
    if (!file_exists($configFile)) {
        throw new Exception('Configuração do banco não encontrada');
    }
    
    $config = json_decode(file_get_contents($configFile), true);
    if (!$config) {
        throw new Exception('Erro ao ler configuração do banco');
    }
    
    sendLog('Conectando ao banco de dados...');
    
    // Conectar com configurações corrigidas
    $pdo = new PDO(
        "mysql:host={$config['host']};dbname={$config['dbname']};charset=utf8mb4",
        $config['username'],
        $config['password'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4",
            PDO::ATTR_TIMEOUT => 60,
            PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => true
        ]
    );
    
    // Configurações MySQL para lidar com datas inválidas
    $pdo->exec("SET SESSION sql_mode = 'ALLOW_INVALID_DATES'");
    $pdo->exec("SET SESSION innodb_lock_wait_timeout = 300");
    $pdo->exec("SET SESSION wait_timeout = 600");
    $pdo->exec("SET SESSION interactive_timeout = 600");
    $pdo->exec("SET autocommit = 0");
    
    sendLog('Conexão estabelecida com configurações otimizadas!', 'success');
    
    // PULAR BACKUP PARA DEBUG MAIS RÁPIDO
    if ($createBackup && $startFrom === 0) {
        sendLog('PULANDO backup para debug mais rápido...');
    }
    
    // Processar arquivo CSV
    $csvFile = $_FILES['csv_file']['tmp_name'];
    $fileSize = filesize($csvFile);
    sendLog('Tamanho do arquivo: ' . formatBytes($fileSize));
    
    // Abrir arquivo
    $handle = fopen($csvFile, 'r');
    if (!$handle) {
        throw new Exception('Erro ao abrir arquivo CSV');
    }
    
    sendLog('Analisando estrutura do arquivo...');
    
    // Detectar delimitador
    $sampleLine = fgets($handle);
    rewind($handle);
    
    $delimiter = ';'; // Fixo baseado no log anterior
    sendLog("Delimitador fixado: '$delimiter'");
    
    // Ler cabeçalho
    $header = fgetcsv($handle, 0, $delimiter);
    if (!$header) {
        throw new Exception('Erro ao ler cabeçalho do CSV');
    }
    
    // Normalizar cabeçalho
    $header = array_map('trim', $header);
    $headerOriginal = $header;
    
    sendLog('Colunas encontradas: ' . count($header));
    sendLog('DEBUG - Primeiras 10 colunas: ' . implode(', ', array_slice($headerOriginal, 0, 10)));
    
    // Ler primeiro registro para testar
    $firstRow = fgetcsv($handle, 0, $delimiter);
    if (!$firstRow) {
        throw new Exception('Erro ao ler primeira linha de dados');
    }
    
    sendLog('DEBUG - Primeira linha tem ' . count($firstRow) . ' campos');
    sendLog('DEBUG - Primeiros 5 campos: ' . implode(' | ', array_slice($firstRow, 0, 5)));
    
    // Combinar dados de teste
    $dadosTest = array_combine($headerOriginal, $firstRow);
    sendLog('DEBUG - Dados de teste combinados: ' . json_encode(array_slice($dadosTest, 0, 8)));
    
    // Voltar ao início para processamento
    rewind($handle);
    fgetcsv($handle, 0, $delimiter); // Pular cabeçalho
    
    // Mapeamento simplificado para debug
    $mapeamento = [
        'associados' => [
            'nome_ass' => 'nome',
            'data_nasc' => 'nasc',
            'sexo' => 'sexo',
            'rg' => 'rg',
            'cpf' => 'cpf',
            'situação' => 'situacao',
            'escolaridade' => 'escolaridade',
            'est_civil' => 'estadoCivil',
            'email' => 'email',
            'fone_celular' => 'telefone'
        ]
    ];
    
    sendLog('DEBUG - Mapeamento configurado');
    
    // Contar registros RAPIDAMENTE
    $totalRegistros = 50; // LIMITADO para debug
    sendLog("DEBUG - Limitando a {$totalRegistros} registros para teste");
    
    // Inicializar contadores
    $stats = [
        'total' => $totalRegistros,
        'processed' => 0,
        'inserted' => 0,
        'updated' => 0,
        'errors' => 0,
        'skipped' => 0
    ];
    
    sendLog('Iniciando processamento em lotes de DEBUG...');
    
    // INICIAR TRANSAÇÃO PRINCIPAL
    $pdo->beginTransaction();
    sendLog('DEBUG - Transação iniciada');
    
    $linhaProcesada = 0;
    $batchCount = 0;
    $startTime = time();
    
    // Preparar statements SIMPLIFICADOS
    sendLog('DEBUG - Preparando statements...');
    
    $stmtCheckAssociado = $pdo->prepare("SELECT id FROM Associados WHERE cpf = ? LIMIT 1");
    $stmtInsertAssociado = $pdo->prepare("
        INSERT INTO Associados (nome, nasc, sexo, rg, cpf, email, situacao, escolaridade, estadoCivil, telefone) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    sendLog('DEBUG - Statements preparados');
    
    // Processar registros
    while (($row = fgetcsv($handle, 0, $delimiter)) !== false && $linhaProcesada < $totalRegistros) {
        try {
            $batchCount++;
            $linhaProcesada++;
            
            sendLog("DEBUG - Processando linha {$linhaProcesada}");
            
            checkConnection();
            
            // Combinar dados
            if (count($row) !== count($headerOriginal)) {
                sendLog("ERRO - Linha {$linhaProcesada}: {count($row)} campos, esperado " . count($headerOriginal), 'error');
                $stats['errors']++;
                continue;
            }
            
            $dadosOriginais = array_combine($headerOriginal, $row);
            $dadosOriginais = array_map('trim', $dadosOriginais);
            
            sendLog("DEBUG - Dados originais OK, CPF: " . ($dadosOriginais['cpf'] ?? 'VAZIO'));
            
            // Mapear dados
            $dataAssociados = mapearCampos($dadosOriginais, $mapeamento['associados']);
            
            sendLog("DEBUG - Dados mapeados, nome: " . ($dataAssociados['nome'] ?? 'VAZIO'));
            
            // Normalizar dados
            $dataAssociados = normalizarDados($dataAssociados);
            
            sendLog("DEBUG - Dados normalizados, CPF: " . ($dataAssociados['cpf'] ?? 'VAZIO'));
            
            // Validar CPF
            if (empty($dataAssociados['cpf'])) {
                sendLog("DEBUG - CPF vazio, pulando registro", 'warning');
                $stats['skipped']++;
                continue;
            }
            
            // Verificar se associado existe
            sendLog("DEBUG - Verificando se CPF {$dataAssociados['cpf']} existe...");
            $stmtCheckAssociado->execute([$dataAssociados['cpf']]);
            $associadoId = $stmtCheckAssociado->fetchColumn();
            
            if ($associadoId) {
                sendLog("DEBUG - Associado existe com ID: {$associadoId}");
                if ($updateExisting) {
                    $stats['updated']++;
                } else {
                    $stats['skipped']++;
                }
            } else {
                sendLog("DEBUG - Inserindo novo associado...");
                
                // Preparar dados para inserção
                $insertData = [
                    $dataAssociados['nome'] ?: null,
                    !empty($dataAssociados['nasc']) ? formatarData($dataAssociados['nasc']) : null,
                    $dataAssociados['sexo'] ?: null,
                    $dataAssociados['rg'] ?: null,
                    $dataAssociados['cpf'],
                    $dataAssociados['email'] ?: null,
                    $dataAssociados['situacao'] ?: null,
                    $dataAssociados['escolaridade'] ?: null,
                    $dataAssociados['estadoCivil'] ?: null,
                    $dataAssociados['telefone'] ?: null
                ];
                
                sendLog("DEBUG - Dados para inserção: " . json_encode($insertData));
                
                $stmtInsertAssociado->execute($insertData);
                
                $associadoId = $pdo->lastInsertId();
                $stats['inserted']++;
                
                sendLog("DEBUG - Inserido com ID: {$associadoId}");
            }
            
            $stats['processed'] = $linhaProcesada;
            
            // Commit a cada lote pequeno
            if ($batchCount >= $batchSize) {
                sendLog("DEBUG - Fazendo commit do lote...");
                $pdo->commit();
                $pdo->beginTransaction();
                
                // Enviar update de progresso
                sendUpdate([
                    'type' => 'progress',
                    'total' => $stats['total'],
                    'processed' => $stats['processed'],
                    'inserted' => $stats['inserted'],
                    'updated' => $stats['updated'],
                    'errors' => $stats['errors'],
                    'skipped' => $stats['skipped']
                ]);
                
                sendLog("DEBUG - Lote processado: {$batchCount} registros");
                
                $batchCount = 0;
                $startTime = time();
            }
            
        } catch (Exception $e) {
            $stats['errors']++;
            sendLog("ERRO linha {$linhaProcesada}: " . $e->getMessage(), 'error');
            sendLog("ERRO detalhes: " . $e->getFile() . ':' . $e->getLine(), 'error');
            
            try {
                $pdo->rollback();
                $pdo->beginTransaction();
            } catch (Exception $rollbackError) {
                sendLog("ERRO no rollback: " . $rollbackError->getMessage(), 'error');
            }
        }
    }
    
    // Commit final
    sendLog("DEBUG - Commit final...");
    $pdo->commit();
    fclose($handle);
    
    // Enviar resultado final
    sendUpdate([
        'type' => 'complete',
        'total' => $stats['total'],
        'processed' => $stats['processed'],
        'inserted' => $stats['inserted'],
        'updated' => $stats['updated'],
        'errors' => $stats['errors'],
        'skipped' => $stats['skipped']
    ]);
    
    sendLog('DEBUG - Importação finalizada!', 'success');
    sendLog("Resumo: {$stats['inserted']} inseridos, {$stats['updated']} atualizados, {$stats['errors']} erros, {$stats['skipped']} ignorados");
    
} catch (Exception $e) {
    sendUpdate([
        'type' => 'error',
        'message' => $e->getMessage()
    ]);
    sendLog('Erro geral: ' . $e->getMessage(), 'error');
    sendLog('Stack trace: ' . $e->getTraceAsString(), 'error');
}

// === FUNÇÕES AUXILIARES ===

function mapearCampos($dadosOriginais, $mapeamento) {
    $dadosMapeados = [];
    foreach ($mapeamento as $campoOriginal => $campoNovo) {
        $dadosMapeados[$campoNovo] = $dadosOriginais[$campoOriginal] ?? '';
    }
    return $dadosMapeados;
}

function formatBytes($size, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }
    return round($size, $precision) . ' ' . $units[$i];
}

function normalizarDados($data) {
    // Converter sexo
    if (isset($data['sexo'])) {
        $sexo = strtolower(trim($data['sexo']));
        if (in_array($sexo, ['masculino', 'masc', 'm'])) {
            $data['sexo'] = 'M';
        } elseif (in_array($sexo, ['feminino', 'fem', 'f'])) {
            $data['sexo'] = 'F';
        }
    }
    
    // Normalizar CPF e RG - APENAS números
    if (isset($data['cpf'])) {
        $data['cpf'] = preg_replace('/[^0-9]/', '', $data['cpf']);
        // Validar se tem 11 dígitos
        if (strlen($data['cpf']) !== 11) {
            $data['cpf'] = ''; // Invalidar CPF
        }
    }
    if (isset($data['rg'])) {
        $data['rg'] = preg_replace('/[^0-9]/', '', $data['rg']);
    }
    
    // Limpar campos vazios
    foreach ($data as $campo => $valor) {
        if (in_array($valor, ['0000-00-00', 'NULL', 'null', '', '0'])) {
            $data[$campo] = '';
        }
    }
    
    return $data;
}

function formatarData($data) {
    if (empty($data)) return null;
    
    $data = trim($data);
    
    // Se já está no formato MySQL
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        // Verificar se é data válida (não 0000-00-00)
        if ($data === '0000-00-00') return null;
        return $data;
    }
    
    // Formato DD/MM/YYYY
    if (preg_match('/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/', $data, $matches)) {
        return sprintf('%04d-%02d-%02d', $matches[3], $matches[2], $matches[1]);
    }
    
    return null;
}
?>